package Mix;
# (c) 2002 Peter Palfrader <peter@palfrader.org>
# $Id: Mix.pm,v 1.1 2002/05/21 02:22:05 weasel Exp $

=pod
=head1 NAME

Mix - mixmaster 2 client routines

=head1 SYNOPSIS

    use Mix;
    
    my $mix = new Mix;
    
    my @chain = qw{mix1 mix2 mix3};

    $mix->mix_smtpsend(
        message => $message,
        chain   => \@chain
    );
    
    $mix->mix(
        message => $message,
        type    => 'post');

=head1 DESCRIPTION

This package provides methods for creating mixmaster 2 messages.

The mixmaster network allows anonymous sending of mail and usenet
messages.

=cut

use lib '../../../../lib', 'lib', '../lib';
use strict;
use Mix::Keyring;
use Crypt::Random qw{};
use Digest::MD5 qw{};
use Crypt::CBC qw{};
use Crypt::RSA qw{};
use Mail::Internet qw{};
use MIME::Base64 qw{};
use Compress::Zlib qw{};
use Carp qw{carp croak confess};
use vars qw{$VERSION
	$TYPE_INTERMEDIATE $TYPE_FINAL $TYPE_FINAL_PARTIAL };

($VERSION) = '$Revision: 1.1 $' =~ /\s(\d+\.\d+)\s/;
$TYPE_INTERMEDIATE = 0;
$TYPE_FINAL = 1;
$TYPE_FINAL_PARTIAL = 2;

# FIXME should be configurable
my @DEFAULT_CHAIN = qw{* * * *};

my $BEGIN_REMAILER_MESSAGE = '-----BEGIN REMAILER MESSAGE-----';
my $END_REMAILER_MESSAGE = '-----END REMAILER MESSAGE-----';
my $TIMESTAMP_MAGIX = "0000\0";
my $GZIP_MAGIX = pack("CC", 31, 139);
my $SECONDS_PER_DAY = 24 * 60 * 60;


=pod

=over

=item B<new> ()

The constructor. Takes a hash, usually with one argument: I<Keyring>.
I<Keyring> is an object of type C<Mix::Keyring>.

=over

=item Keyring

An object of type C<Mix::Keyring>. The keyring holds the public keys of all
known mixmaster remailers. If it is not given, it is created with no parameters.
See the C<Mix::Keyring> documentation for the files searched in that case.

=back

=cut

sub new {
	my ($class, %params) = @_;
	my $self = {};
	bless $self, $class;

	if (defined $params{'Keyring'}) {
		$self->{'Keyring'} = $params{'Keyring'};
	} else {
		$self->{'Keyring'} = new Mix::Keyring ();
	};
	return $self;
};

# Encrypts $plaintext with the given Initialization Vector (iv) and key using DES EDE.
# note that the length of plaintext needs to be a multiple of the blocksize (8).
sub des_encrypt ($$$) {
	my ($plaintext, $iv, $key) = @_;
	carp("plaintext length is not a multiple of 8 (".length($plaintext).")!") unless (length($plaintext) % 8 == 0);
	carp("initialization vector is not 8 bytes long!") unless (length($iv) == 8);
	carp("key is not 24 bytes long!") unless (length($key) == 24);

	
	my $cipher = new Crypt::CBC( {
		'cipher'		=> 'Crypt::DES_EDE3',
		'key'			=> $key,
		'regenerate_key'	=> 0,
		'iv'			=> $iv,
		'prepend_iv'		=> 0 });
	my $ciphertext = $cipher->encrypt($plaintext);
	if (length($ciphertext) != length($plaintext) + 8) {
		carp ("did not expect that length (".length($ciphertext)." != ".length($plaintext)." + 8)");
		return $ciphertext;
	} else {
		return substr ($ciphertext, 0, -8); # Cut off last 8 bytes, they are not needed anyway
	}
}

# Decrypts $ciphertext with the given Initialization Vector (iv) and key using DES EDE.
# note that the length of ciphertext needs to be a multiple of the blocksize (8).
sub des_decrypt ($$$) {
	my ($ciphertext, $iv, $key) = @_;
	carp("ciphertext length is not a multiple of 8 (".length($ciphertext).")!") unless (length($ciphertext) % 8 == 0);
	carp("initialization vector is not 8 bytes long!") unless (length($iv) == 8);
	carp("key is not 24 bytes long!") unless (length($key) == 24);

	
	my $cipher = new Crypt::CBC( {
		'cipher'		=> 'Crypt::DES_EDE3',
		'key'			=> $key,
		'regenerate_key'	=> 0,
		'iv'			=> $iv,
		'prepend_iv'		=> 0 });
	my $plaintext = $cipher->decrypt($ciphertext . "\0" x 8);

	if (length($ciphertext) != length($plaintext)) {
		carp ("did not expect that length (".length($ciphertext)." != ".length($plaintext)." + 8)");
	};
	return $plaintext;
}

# Encrypt plaintext with the key using RSA (PKCS1v15)
sub rsa_encrypt ($$) {
	my ($plaintext, $key) = @_;
	my $rsa = new Crypt::RSA ( ES => 'PKCS1v15' );
	($plaintext) = $plaintext =~ /^(.*)$/s; # untaint plaintext (FIXME)
	return $rsa->encrypt (
		Message	=> $plaintext,
		Key	=> $key
	) or confess $rsa->errstr();
};

# Decrypt plaintext with the key using RSA (PKCS1v15)
sub rsa_decrypt ($$) {
	my ($ciphertext, $key) = @_;
	my $rsa = new Crypt::RSA ( ES => 'PKCS1v15' );
	($ciphertext) = $ciphertext =~ /^(.*)$/s; # untaint ciphertext (FIXME)
	return $rsa->decrypt (
		Cyphertext	=> $ciphertext,
		Key	        => $key
	) or confess $rsa->errstr();
};

# Provides $length octets of strong randomness
sub make_random ($) {
	my ($length) = @_;

	return '' if ($length == 0);
	return Crypt::Random::makerandom_octet(Length => $length);
}

# Provides (size - length(data)) octets of weak randomness
sub random_padding ($$) {
	my ($data, $size) = @_;
	my $length = length $data;
	# untaint length (FIXME)
	($length) = $length =~ /^([0-9]+)$/ or confess ("Unexpected characters in length() result: '$length'");

	if ($length > $size) {
		confess ("data is already longer than size to pad to ($length > $size)");
		return '';
	};
	
	return '' if ($size == $length);
	return Crypt::Random::makerandom_octet(Length => $size - $length, Strength => 0);
}

# Proved (size - length(data)) octets of zeroes
sub zero_padding ($$) {
	my ($data, $size) = @_;
	my $length = length $data;
	# untaint length
	($length) = $length =~ /^([0-9]+)$/ or confess ("Unexpected characters in length() result: '$length'");

	if ($length > $size) {
		confess ("data is already longer than size to pad to");
		return '';
	};
	
	return ' ' x  ($size - $length);
}

=pod

=item B<build_message> (%args)

Build mixmaster packets of I<message> chained through the list of remailers
given in I<chain>. args keys:

=over

=item I<type> (optional)

Type of the message, either C<mail> (the default), C<post> or C<dummy>.

=item I<message>

The message to send through the remailers. Required unless I<type> is C<dummy>.
I<message> needs to be in RFC2822 or in RFC1036 format.

=item I<chain> (optional)

Reference to an array of remailer names through which the messag should be
sent. An asterisk C<*> may be used to chose a random remailer. If not given, a
default chain is used (usually something like *,*,*,*).

=item <copies>

Number of copies to be send. If random hops are used in the chain, it is
chosen independetly for each copy (as is also done if the message is too
large to be send with one packet). The last hop is the same for all messages.

=back

Returns a list of mixmaster packets, not yet suitable for sending through mail.
Also see I<make_mail>.

=cut

sub build_message ( $% ) {
	my ($self, %args) = @_;
	# my ($self, $message, @chain) = @_;

	my $message_type = $args{'type'} || 'mail';
	my $message = $args{'message'};
	my $copies = defined $args{'copies'} ? $args{'copies'} : 1;
	my @chain = @{ $args{'chain'} } ? @{ $args{'chain'} } : @DEFAULT_CHAIN;

	$message = '' if ($message_type eq 'dummy');
	croak "error: no message" unless (defined $message);
	croak "error: no chain" unless (scalar @chain);


	my $payload = '';
	if ($message_type eq 'dummy') {
		$payload .= pack ("C", 1); # number of destination fields
		$payload .= pack ("a80", "null");
	elsif ($message_type eq 'post') {
		$payload .= pack ("C", 1); # number of destination fields
		$payload .= pack ("a80", "post:");
	} else {
		$payload .= pack ("C", 0); # number of destination fields
	};
	$payload .= pack ("C", 0); # number of header fields


	my $user_data_section = '';
	$user_data_section .= "##\r\n";
	$user_data_section .= $message;

	if (length($user_data_section) > 10236 - (length $payload)) {
		$user_data_section = Compress::Zlib::memGzip($user_data_section);
	};


	$payload .= $user_data_section;


	my @packets;

	my $number_of_packets = int ((length $payload) / 10236) + 1;
	my $message_id = make_random(16);

	# FIXME random hop: the last hop needs to be the same for all
	# copies. if the last of a chain or multipacket message is random
	# chose it now
	
	for (my $copynum = 0; $copynum < $copies; $copynum++ ) {
		for (my $packetnum = 0; $packetnum < $number_of_packets; $packetnum++ ) {
			my $body = substr($payload, 10236 * $packetnum, 10236);
			$body = pack ("V", length $body) . $body;
			$body .= random_padding($body, 10240);

			my @heads;
			my $next_hop_address;
			for my $hop (reverse @chain) {
				my $key = make_random(24);
				my $iv = make_random(8);

				$body = des_encrypt($body, $iv, $key);

				my $crypthead = make_random(16);
				$crypthead .= $key;
				if (scalar @heads == 0) {
					if ($number_of_packets == 1) {
						$crypthead .= pack ("C", $TYPE_FINAL);
						$crypthead .= $message_id;
						$crypthead .= $iv;
					} else {
						$crypthead .= pack ("C", $TYPE_FINAL_PARTIAL);
						$crypthead .= pack ("C", $packetnum + 1);
						$crypthead .= pack ("C", $number_of_packets);
						$crypthead .= $message_id;
						$crypthead .= $iv;
					}
				} else {
					my @ivs = map { make_random(8) } (1 .. 18);
					push @ivs, $iv;
					for (my $i = 0; $i < scalar @heads; $i++) {
						$heads[$i] = des_encrypt($heads[$i], $ivs[$i], $key);
					};

					$crypthead .= pack ("C", $TYPE_INTERMEDIATE);
					$crypthead .= join '', @ivs;
					$crypthead .= $next_hop_address;
					$crypthead .= zero_padding($next_hop_address, 80);
				}
				$crypthead .= $TIMESTAMP_MAGIX;
				$crypthead .= pack ("v", time / $SECONDS_PER_DAY);
				$crypthead .= Digest::MD5::md5($crypthead);
				$crypthead .= random_padding($crypthead, 328);

				my $session_key = make_random(24);
				my $session_iv = make_random(8);
				$crypthead = des_encrypt($crypthead, $session_iv, $session_key);

				# FIXME random hop
				# FIXME new keyhandling
				my $recipient = $self->{'Keyring'}->{'remailers'}->{$hop};
				croak ("No key for remailer $hop") unless (defined $recipient->{'V2Key'}->{'Key'});
				my $encrypted_session_key = rsa_encrypt($session_key, $recipient->{'V2Key'}->{'Key'});

				my $head;
				$head = $recipient->{'V2Key'}->get_keyid();
				$head .= pack ("C", length($encrypted_session_key)); # FIXME (check size!)
				$head .= $encrypted_session_key;
				$head .= $session_iv;
				$head .= $crypthead;
				$head .= random_padding($head, 512);

				unshift @heads, $head;
				$next_hop_address = $recipient->{'Address'};
			}
			my $head = join '', @heads;
			$head .= random_padding($head, 10240);
			my $packet;
			$packet = $head . $body;
			carp ("packet size is not 20480") unless (length $packet == 20480);
			push @packets, {
				'data' => $packet,
				'to' => $next_hop_address
				};
		};
	};
	return @packets;
}

=pod

=item B<make_mail> (%args)

Takes a packet as produced by I<build_message> and returns a
valid C<Mail::Internet> object to be send to the first hop.

=over

=item packet

The packet to encode.

=back

=cut
sub make_mail ($%) {
	my ($self, %args) = @_;

	my $packet = $args{'packet'}
		or croak "error: no packet";

	my $digest = Digest::MD5::md5($packet->{'data'});
	$digest = MIME::Base64::encode($digest);

	my $message = MIME::Base64::encode($packet->{'data'});
	$message =~ tr|A-Za-z0-9+=/||cd;
	my @message = map { $_ . "\n" } grep { $_ } split /(.{1,40})/, $message;

	my $header = new Mail::Header;
	$header->add ('To', $packet->{'to'} );

	my @body = (
		"::\n",
		"Remailer-Type: Mixmaster 2.9-pmix_$VERSION\n",
		"\n",
		$BEGIN_REMAILER_MESSAGE."\n",
		"20480\n",
		$digest,
		@message,
		$END_REMAILER_MESSAGE."\n" );

	my $mail = new Mail::Internet (
		Header => $header,
		Body => \@body
	);
}

=pod

=item B<mix> (%args)

Combines both I<build_message> and I<make_mail>. Returns a list of
C<Mail::Internet> objects. Argument keys are documented at I<build_message>.

=cut

sub mix ($%) {
	my ($self, %args) = @_;

	my @packets = $self->build_message(%args);
	my @mails = map { $self->make_mail( 'packet' => $_) } @packets;
	#for my $packet (@packets) {
	#	my $mail = $self->make_mail($packet);
	#	push @mails, $mail;
	#};
	return @mails;
};

=pod

=item B<mix_smtpsend> (%args)

Similar to I<mix> but additionaly sends all mails using C<Mail::Internet-E<gt>send>
(which uses C<Mail::Mailer>). Argument keys are documented at I<build_message>.

=cut

sub mix_send ($%) {
	my ($self, %args) = @_;
	
	my @mails = $self->mix(%args);
	for my $mail (@mails) {
		$mail->send();
	};
	return @mails;
};

=pod

=item B<mix_smtpsend> (%args)

Similar to I<mix> but additionaly sends all mails using C<Mail::Internet-E<gt>smtpsend>
(which uses C<Net::SMTP>). Argument keys are documented at I<build_message>.

=cut

sub mix_smtpsend ($%) {
	my ($self, %args) = @_;
	
	my @mails = $self->mix(%args);
	for my $mail (@mails) {
		$mail->smtpsend();
	};
	return @mails;
};

=pod

=item B<is_mix_message> (I<mail>)

This method determines whether I<mail> (an object of type C<Mail::Internet>) seems
to be a valid mixmaster message. In this case it returns 1. Otherwise returns 0.

=cut

sub is_mix_message($$) {
	my ($self, $mail) = @_;

	my $lines = $mail->body();

	return 0 unless (scalar @$lines >= 5);
	my $lineno = 0;
	my $line;

	chomp($line = $lines->[$lineno++]);
	return 0 unless ($line eq '::');

	chomp($line = $lines->[$lineno++]);
	return 0 unless ($line =~ /^Remailer-Type: Mixmaster 2\./);

	chomp($line = $lines->[$lineno++]);
	return 0 unless ($line eq '');

	chomp($line = $lines->[$lineno++]);
	return 0 unless ($line eq $BEGIN_REMAILER_MESSAGE);

	for (; $lineno < scalar @$lines; $lineno++) {
		chomp($line = $lines->[$lineno]);
		return 1 if ($line eq $END_REMAILER_MESSAGE);
	};
	return 0;	
}

=pod

=item B<decode_mail> (I<mail>)

Decodes I<mail> (object of type C<Mail::Internet>) and returns the
mixmaster packet transported in mail.
Returns undef if a length or message digest mismatch is encountered.

=cut

sub decode_mail ($$) {
	my ($self, $mail) = @_;

	my $from = $mail->head()->get('From') || '<unkown>';

	my @msg;
	my $started = 0;
	my $lines = $mail->body();
	for my $line (@$lines) {
		chomp $line;
		unless ($started) {
			if ($line eq $BEGIN_REMAILER_MESSAGE) {
				$started = 1;
			};
			next;
		};
		last if ($line eq $END_REMAILER_MESSAGE);
		push @msg, $line;
	};
	return undef unless $started;
	
	my $size = shift @msg;
	my $digest = shift @msg;
	my $packet = join '', @msg;
	
	$digest = MIME::Base64::decode($digest);
	$packet = MIME::Base64::decode($packet);

	if ($size =~ /[^0-9]/) {
		carp ("Invalid type 2 message from '$from': unexpected characters in size\n");
		return undef;
	}
	if ($size != 20480) {
		# may be removed
		carp ("Invalid type 2 message from '$from': size is not 20480\n");
		return undef;
	};
	if ($size != length($packet)) {
		carp ("Invalid type 2 message from '$from': size mismatch\n");
		return undef;
	};
	if ($digest ne Digest::MD5::md5($packet)) {
		carp ("Invalid type 2 message from '$from': digest mismatch\n");
		return undef;
	};
	return {
		to => $mail->head()->get('To'),
		from => $from,
		data => $packet
	};
};

=pod

=item B<decode_packet> (I<packet>)

Decodes and decrypts the mixmaster packet in I<packet>.
Returns a list of 2 entries, the packet type and additional data which type
depends on the packat type. Returns undef if the private mixmaster key is
not on the keyring or some other error is encountered (wrong digests etc).

Possible packet types are:

=over

=item INTERMEDIATE

returns: ($TYPE_INTERMEDIATE, I<packet>). I<packet> is a mixmaster packet which
should be send to the next remailer in the packets chain. Use I<make_mail> to
build a mail message from the packet.

=item FINAL

returns: ($TYPE_FINAL, I<payload>). I<payload> is the data from the mixmaster
packet. Payloads are handled in I<decode_payload>.

=item FINAL_PARTIAL

returns: ($TYPE_FINAL_PARTIAL, I<payload_part>). I<payload_part> is one
part of a final message. If all parts of the message have been received, they
should be joined (I<decode_payload>) and sent to the final destination.

=back

=cut

sub decode_packet ($$) {
	my ($self, $packet) = @_;

	my $from = $packet->{'from'} || '<unknown>';
	my $data = $packet->{'data'};


	

	unless (length($data) == 20480) {
		# if this is removed, please do some other length checks below
		carp ("Invalid type 2 message from '$from': packet data is not 20480 bytes.\n");
		return undef;
	};
	


	my ($myheader, $header, $body);
	($myheader, $header, $body) = unpack("a512 a9728 a*", $data);
	unless (length($body) == 10240) {
		confess ("ASSERTION FAILED: body should really be 10240 bytes long here, is: '".length($body)."'");
	};




	my ($public_keyid, $length, $encrypted_session_key, $session_key, $session_iv, $crypthead, $rsa_key);
	($public_keyid, $length, $myheader) = unpack("a16 C a*", $myheader);

	$public_keyid = unpack ("H*", $public_keyid);
	$rsa_key = $self->{'Keyring'}->{'bykeyid'}->{$public_keyid}->{'secretkey'};
	unless (defined $rsa_key) {
		carp ("Invalid type 2 message from '$from': do not have private key $public_keyid\n");
		return undef;
	};
	
	if ($length != 128) {
		# may be removed
		carp ("Invalid type 2 message from '$from': length of RSA encrypted data is not 128\n");
		return undef;
	};
	($encrypted_session_key, $session_iv, $crypthead) = unpack("a$length a8 a328", $myheader);
	$session_key = rsa_decrypt($encrypted_session_key, $rsa_key);
	$crypthead = des_decrypt($crypthead, $session_iv, $session_key);



	my ($origcrypthead, $digest_length);
	my ($packet_id, $key, $type);
	my (@ivs, $next_hop_address, $iv, $message_id, $chunk_number, $total_chunks);
	my ($timestamp_magix, $timestamp, $digest);


	($packet_id, $key, $type, $crypthead) = unpack ("a16 a24 a1 a*", $crypthead);
	$digest_length = 16 + 24 + 1;

	# FIXME packet id  dup check

	if ($type == $TYPE_INTERMEDIATE) {
		my $tmp;
		($tmp, $next_hop_address, $crypthead) = unpack ("a152 a80 a*", $crypthead);
		$digest_length += 152 + 80;
		@ivs = unpack("a8" x 19, $tmp);
		$iv = $ivs[18];
		$next_hop_address =~ s/\0*$//;
	} elsif ($type == $TYPE_FINAL) {
		($message_id, $iv, $crypthead) = unpack ("a16 a8 a*", $crypthead);
		$digest_length += 16 + 8;
	} elsif ($type == $TYPE_FINAL_PARTIAL) {
		($chunk_number, $total_chunks, $message_id, $iv, $crypthead) = unpack ("a1 a1 a16 a8 a*", $crypthead);
		$digest_length += 1 + 1 + 16 + 8;
	} else {
		carp ("Invalid type 2 message from '$from': unkown packet type $type\n");
		return undef;
	};

	$timestamp_magix = unpack ("a5", $crypthead);
	if ($timestamp_magix == $TIMESTAMP_MAGIX) {
		($timestamp_magix, $timestamp, $crypthead) = unpack ("a5 a2 a*", $crypthead);
		$digest_length += 5 + 2;
	};
	($digest) = unpack ("a16", $crypthead);
	if ($digest ne Digest::MD5::md5 (unpack("a $digest_length", $origcrypthead))) {
		carp ("Invalid type 2 message from '$from': header digest mismatch\n");
		return undef;
	};
	
	

	$body = des_decrypt($body, $iv, $key);

	if ($type == $TYPE_INTERMEDIATE) {
		my @heads = unpack ("a512" x 19, $header);
		for (my $i=0; $i<19; $i++) {
			$heads[$i] = des_decrypt($heads[$i], $ivs[$i], $key);
		};
		$header = join '', @heads;
		$header .= random_padding($header, 10240);

		$packet = $header . $body;

		return (
			$TYPE_INTERMEDIATE,
			{
				'data' => $packet,
				'to' => $next_hop_address
			}
		);
	};

	my ($body_length);

	($body_length, $body) = unpack ("V a*", $body);

	if ($body_length <= 0 || $body_length > length($body)) {
		carp ("Invalid type 2 message from '$from': Body length has impossible valued $body_length\n");
		return undef;
	};

	$body = unpack ("a$body_length", $body);

	if ($type == $TYPE_FINAL) {
		return (
			$TYPE_FINAL,
			{
				message_id => $message_id,
				chunk_number => 1,
				total_chunks => 1,
				payload => $body
			}
		);
	} else {
		return (
			$TYPE_FINAL_PARTIAL,
			{
				message_id => $message_id,
				chunk_number => $chunk_number,
				total_chunks => $total_chunks,
				payload => $body
			}
		);
	};
};


=pod

=item B<decode_payload> (I<payload>)

=item B<decode_payload> (I<payload_parts>)

Takes one payload or a list of payload_parts are processes them.

Returns:

=over

=item ('mail', C<Mail::Internet>)

if the message is an internet email message.

=item ('post', C<Mail::Internet>)

if the message is a usenet news posting.

=item ('dummy', undef)

if the message is a dummy (null:) message.

=back

=cut

sub decode_payload ($@) {
	my ($self, @payloads) = @_;

	return undef unless (scalar @payloads);
	my $msgid = $payloads[0]->{'message_id'};

	# FIXME message id  dup check



	my $total_chunks = $payloads[0]->{'total_chunks'};
	unless ($total_chunks == scalar @payloads) {
		carp ("Not all chunks passed to decode_payload.");
		return undef;
	};
	

	my %by_number;
	for my $payload (@payloads) {
		unless ($msgid eq $payload->{'message_id'}) {
			carp ("Payloads passed to decode_payload are not from the same message.");
			return undef;
		};
		if (defined $by_number{$payload->{'chunk_number'}}) {
			carp ("Chunk ".$payload->{'chunk_number'}." encountered more than once.");
			return undef;
		};
		 $by_number{$payload->{'chunk_number'}} = $payload;
	};

	my $payload = '';
	for (my $chunk_number = 1; $chunk_number <= $total_chunks; $chunk_number++) {
		$payload .= $by_number{$chunk_number}->{'payload'};
	};

	my @recipients;
	my @newsgroups;
	my @headerlines;
	my $message_type = 'mail';
	
	my ($number_of_destinations, $number_of_headerlines);
	# FIXME check that payload is long enough
	($number_of_destinations, $payload) = unpack ("a a*", $payload);

	for (my $i; $i < $number_of_destinations; $i++) {
		# FIXME check that payload is long enough
		my $destline;
		($destline, $payload) = unpack ("a80 a*", $payload);

		$destline =~ s/\0*$//;
		# FIXME FIXME parse destline
		if ($destline =~ /^null:/ ) {
			return ('dummy', undef);
		} elsif ($destline =~ /^post:/ ) {
			my ($group) =~ /^post:\s*(.*)/;
			push @newsgroups, $group if defined $group;
			$message_type = 'post';
		} else {
			push @recipients, $destline;
		};
	};

	if (scalar @recipients && scalar @newsgroups) {
		# FIXME now what?
	};
	
	# FIXME check that payload is long enough
	($number_of_headerlines, $payload) = unpack ("a a*", $payload);

	for (my $i; $i < $number_of_headerlines; $i++) {
		# FIXME check that payload is long enough
		my $headerline;
		($headerline, $payload) = unpack ("a80 a*", $payload);

		$headerline =~ s/\0*$//;
		push @headerlines, $headerline
	};

	# FIXME check that payload is long enough
	
	my $gzip_magix = unpack ("a2", $payload);
	if ($gzip_magix eq $GZIP_MAGIX) {
		($gzip_magix, $payload) = unpack ("a2 a*", $payload);
		$payload = Compress::Zlib::memGunzip($payload);
		# FIXME any errors uncompressing?
	};

	my @lines;
	if (scalar @newsgroups) {
		push @lines, "Newsgroups: " . (join ', ', @newsgroups) . "\n";
	};
	if (scalar @recipients) {
		push @lines, "To " . (join ', ', @recipients) . "\n";
	};
	
	push @lines, (map { $_ . "\n" } split (/\n/, $payload));
	
	my $msg = new Mail::Internet (\@lines);
	
	return ($message_type, $msg);
}

=pod

=back

=head1 AUTHOR

Peter Palfrader E<lt>peter@palfrader.orgE<gt>

=head1 BUGS

Please report them to the author.

=head1 SEE ALSO

=over

=item Mixmaster Protocol, Version 2 <draft-moeller-v2-01.txt>

=item mix(1)

=item news:alt.privacy.anon-server

=item Mix
        
=back

=cut

