package Mix::Capabilities;
# (c) 2002 Peter Palfrader <peter@palfrader.org>
# $Id: Capabilities.pm,v 1.1 2002/05/21 02:22:05 weasel Exp $

=pod
=head1 NAME

Mix::Capabilities - Mixmaster remailer capabilities

=head1 SYNOPSIS

 use Mix::Capabilities;

 $caps = new Mix::Capabilities;
 $caps->mix2caps( caps => 'CNm' );
 
 $capstring = $caps->get_mix2_capstring();

=head1 DESCRIPTION

This package provides basic capability handling functionality for
Mix::Remailer.

=cut

use strict;
use lib '../../../../lib', 'lib', '../lib';
use Carp qw{carp croak confess};
use vars qw{$VERSION};

($VERSION) = '$Revision: 1.1 $' =~ /\s(\d+\.\d+)\s/;


=pod

=over

=item B<new> (%args)

The constructor. %args has the following optional keys: C<mix2>,
C<middle>, C<compress>, C<post>, C<postm2n>, C<postdirect> and
C<mix2caps>. If C<mix2caps> it is passed on to I<parse_mix2_caps>().

=cut

sub new {
	my ($class, %args) = @_;
	my $self = {};

	$self->{'mix'} = $args{'mix2'};
	$self->{'middle'} = $args{'middle'};
	$self->{'compress'} = $args{'compress'};
	$self->{'post'} = $args{'post'};
	$self->{'postm2n'} = $args{'postm2n'};
	$self->{'postdirect'} = $args{'postdirect'};

	bless $self, $class;

	$self->mix2caps(Caps => $args{'mix2caps'})
		if defined $args{'mix2caps'};
	return $self;
};

=item B<mix2caps> (%args)

Parses mix2 capability strings. Keys:

=over 

=item B<Caps>

The capability string to parse. An example I<mix2 capstring> might look like
this: C<CNm>. If it is given, mix2 is also set. If it's undefined nothing is
done.

=back

=cut
sub mix2caps($) {
	my ($self, %args) = @_;

	$self->{'mix2'} = defined $args{'Caps'};
	return unless defined $args{'Caps'};
	
	$self->{'compress'} = $args{'Caps'} =~ /C/;
	$self->{'middle'} = $args{'Caps'} =~ /M/;
	$self->{'post'} = $args{'Caps'} =~ /N/;
	$self->{'postm2n'} = $args{'Caps'} =~ /m/;
	$self->{'postdirect'} = $args{'Caps'} =~ /p/;
};


=item B<get_mix2_capstring> ()

Returns the mix2 capstring.

=cut
sub get_mix2_capstring($) {
	my ($self) = @_;

	return undef
		unless defined $self->{'mix2'};
	my $result = '';
	$result .= 'C' if $self->{'compress'};
	$result .= 'M' if $self->{'middle'};
	$result .= 'N' if $self->{'post'};
	$result .= 'm' if ($self->{'post'} && $self->{'postm2n'});
	$result .= 'p' if ($self->{'post'} && $self->{'postdirect'});

	return $result;
};


=pod

=back

=head1 AUTHOR

Peter Palfrader E<lt>peter@palfrader.orgE<gt>

=head1 BUGS
 
Please report them to the author.

=head1 SEE ALSO

Mixmaster Protocol, Version 2 <draft-moeller-v2-01.txt>, 
mix(1),
news:alt.privacy.anon-server,
Mix

=cut
