package Mix::Keyring;
# (c) 2002 Peter Palfrader <peter@palfrader.org>
# $Id: Keyring.pm,v 1.1 2002/05/21 02:22:05 weasel Exp $

=pod
=head1 NAME

Mix::Keyring - handle the mixmaster keyring

=head1 SYNOPSIS

FIXME doc

=head1 DESCRIPTION

FIXME doc

=cut

use strict;
use lib '../../../../lib', 'lib', '../lib';
use Mix::Remailer;
use Digest::MD5 qw{};
use MIME::Base64 qw{};
use Math::Pari qw(PARI);
use Crypt::RSA;
use Crypt::RSA::Key;
use Carp qw{carp croak confess};
use vars qw{$VERSION};

($VERSION) = '$Revision: 1.1 $' =~ /\s(\d+\.\d+)\s/;


my @MIXRINGS = (
	$ENV{'HOME'}.'/.mix/pubring.mix',
	$ENV{'HOME'}.'/Mix/pubring.mix',
	'/var/lib/mix/pubring.mix',
	'/etc/mix/pubring.mix'
	);


=pod

=over

=item B<new> ()

The constructor. Takes a hash, usually with one argument: I<Filename>.
I<Filename> indicates a file from which the public mixmaster keyring should
be read.

=over

=item Filename

Name of the file that contains the public mixmaster keyring. If not specified
the following filenames are tried in this order:

=over

=item $HOME/.mix/pubring.mix

=item $HOME/Mix/pubring.mix

=item /var/lib/mix/pubring.mix

=item /etc/mix/pubring.mix

=back

For importing the keyring, I<read_file> is called.

=back

=cut

sub new {
	my ($class, %params) = @_;
	my $self = {};
	bless $self, $class;

	if (defined $params{'Filename'}) {
		$self->read_file($params{'Filename'});
	} else {
		for my $filename (@MIXRINGS) {
			if (-e $filename) {
				$self->read_file($filename);
				last;
			};
		};
	};
	unless (scalar keys %{$self->{'remailers'}}) {
		warn ("No public remailer keys found\n");
	}
	return $self;
};

=pod

=item B<read_ring> (I<ARG>)

Reads all keys from the keyring passed as ARG and adds it to the keyring object.

=over

I<ARG> may be either a file descriptor (reference to a GLOB) or a reference to
an array.

=back

=cut

sub read_ring($@) {
	my ($self, $arg) = @_;

	my $lines;
	if (ref($arg) eq 'ARRAY') {
		$lines = $arg;
	} elsif (defined fileno($arg)) {
		@$lines = <$arg>;
	};

	my %capabilities;
	my %keys;

	my $key;
	for(@$lines) {
		chomp;
		if (/^(\S+) \s+ (\S+\@\S+) \s+ ([0-9a-fA-F]{32}) \s+ (\S+) (?:\s+)? (\S+)?$/x) {
			my $keyid = get_capline_keyid( $_ );
			#my $keyid = Mix::Crypt::Remailer::get_capline_keyid( $_ );
			$capabilities{$keyid} = $_ if defined $keyid;
		} elsif (/^-----Begin Mix Key-----$/) {
			@$key = ();
		} elsif (/^-----End Mix Key-----$/) {
			my $keyid = get_keyblock_keyid( $key );
			#my $keyid = Mix::Crypt::Remailer::get_keyblock_keyid( $key );
			$keys{$keyid} = $key if defined $keyid;
			$key = undef;
		} elsif (/^$/) {
		} else {
			if (defined $key) {
				push @$key, $_;
			} else {
				warn ("Unexpected line '$_'\n");
			}
		}
	}

	my %unique;
	for my $keyid ( grep { !$unique{$_}++ } (keys %capabilities, keys %keys)) {
		unless (defined ($keys{$keyid})) {
			carp ("Key '$keyid' has capline but no key");
			next;
		};
		unless (defined ($capabilities{$keyid})) {
			carp ("Key '$keyid' has key but no capline");
			next;
		};

		$self->{'by-key-id'}->{$keyid} = new Mix::Remailer (
			Keyblock => $keys{$keyid},
			Capline => $capabilities{$keyid}
		);
		
	};

	for my $keyid (keys %{$self->{'by-key-id'}}) {
		$self->{'remailers'}->{$self->{'by-key-id'}->{$keyid}->{'Identifier'}} = $self->{'by-key-id'}->{$keyid};
	}
}

=pod

=item B<read_file> (I<filename>)

Opens the file given by I<filename> and calls I<read_ring> to import all keys from the given file.

=cut


sub read_file($$) {
	my ($self, $filename) = @_;
	open (KEYRING, $filename) or croak ("Cannot open public keyring '$filename': $!");
	$self->read_ring(*KEYRING);
	close(KEYRING);
};


=item B<get_type2list> ()

Returns a string with the type2.list.

=cut

sub get_type2list($) {
	my ($self) = @_;

	my $result = '';
	for my $remailer (sort keys %{$self->{'remailers'}}) {
		$result .= $self->{'remailers'}->{$remailer}->get_type2_line();
	};

	return $result;
};

=item B<gen_key> (%args)

Generates a new mixmaster public/private keypair.

=cut

# FIXME
# FIXME
# FIXME
sub gen_key ($%) {
	my ($self, %args) = @_;
	
	my $rsa_key = new Crypt::RSA::Key;
	my ($public, $private) = $rsa_key->generate(
		Size	=> 1024
	) or carp $rsa_key->errstr();

	
};

=pod

=back

=head1 AUTHOR

Peter Palfrader E<lt>peter@palfrader.orgE<gt>

=head1 FILES

I<$HOME/.mix/pubring.mix>, I<$HOME/Mix/pubring.mix>, I</var/lib/mix/pubring.mix>,
and I</etc/mix/pubring.mix> are tried in that order if no I<Filename>
argument is passed to I<new>.

=head1 BUGS
 
Please report them to the author.

=head1 SEE ALSO

=over

=item Mixmaster Protocol, Version 2 <draft-moeller-v2-01.txt>

=item mix(1)

=item news:alt.privacy.anon-server

=item Mix

=back

=cut
1;
