package Mix::Remailer;
# (c) 2002 Peter Palfrader <peter@palfrader.org>
# $Id: Remailer.pm,v 1.1 2002/05/21 02:22:05 weasel Exp $

=pod
=head1 NAME

Mix::Remailer - Information about one remailer (key, conf, ..)

=head1 SYNOPSIS

FIXME docs

=head1 DESCRIPTION

FIXME docs

=cut

use strict;
use lib '../../../../lib', 'lib', '../lib';
use Mix::V2Key;
use Mix::Capabilities;
use Digest::MD5 qw{};
use MIME::Base64 qw{};
use Math::Pari qw(PARI);
use Carp qw{carp croak confess};


BEGIN {
	use Exporter ();
	our ($VERSION, @ISA, @EXPORT, @EXPORT_OK, %EXPORT_TAGS);

	($VERSION) = '$Revision: 1.1 $' =~ /\s(\d+\.\d+)\s/;
	@ISA = qw{Exporter};
	@EXPORT = qw{ &get_capline_keyid &get_keyblock_keyid };
	
	@EXPORT_OK = qw{&get_capline_keyid &get_keyblock_keyid };
};



=pod

=over

=item B<new> (%args)

=over

=item Keyblock

The public keyblock.

=item Capline

The capability line.

=back

=cut

sub new {
	my ($class, %args) = @_;
	my $self = {};

	bless $self, $class;

	$self->{'V2Key'} = new Mix::V2Key ( Keyblock => $args{'Keyblock'} );
	$self->set_from_capline( $args{'Capline'} );
	return $self;
};


=item B<get_type2_line> ()

Returns the cap line used in type2.list files.

=cut
sub get_type2_line($) {
	my ($self) = @_;

	my $result = sprintf("%s %s %s %s %s\n",
		$self->{'Identifier'},
		$self->{'Address'},
		$self->{'V2Key'}->get_keyid_hex(),
		$self->{'Version'},
		$self->{'Capabilities'}->get_mix2_capstring());
	return $result;
};


=item B<set_from_capline> (I<capline>)

Sets Identifier, Address, Version and Capabilities from a I<capline> like
 marvin mix@marvin 21f7d8f2f12c29bf5d97a6d3435241b9 2.9b33 CNm

The keyid is checked against the stored key for this remailer.

=cut

sub set_from_capline($) {
	my ($self, $capline) = @_;
	
	unless (defined $capline) {
		carp ('capline not defined');
		return undef;
	};
	
	my ($identifier, $address, $keyid, $version, $capabilities);

	unless (($identifier, $address, $keyid, $version, $capabilities) = 
		( $capline =~ /^(\S+) \s+ (\S+\@\S+) \s+ ([0-9a-fA-F]{32}) \s+ (\S+) (?:\s+)? (\S+)?$/x ) ) {
		carp ("could not parse capline");
		return undef;
	};

	$keyid = lc $keyid;
	unless ($keyid eq ($self->{'V2Key'}->get_keyid_hex())) {
		carp ("keyid does not match key");
		return undef;
	};

	$self->{'Identifier'} = $identifier;
	$self->{'Address'} = $address;
	$self->{'Version'} = $version;
	$self->{'Capabilities'} = new Mix::Capabilities ( mix2caps => $capabilities || "" );

	return $keyid;
};



=item B<get_capline_keyid> (I<capline>)

Returns the keyid (in hex) of a mix2 capline like
 marvin mix@marvin 21f7d8f2f12c29bf5d97a6d3435241b9 2.9b33 CNm

This is a "static" method, i.e. use it as
C<Mix::Remailer::get_capline_keyid($capline)> and not as
C<$remailer->get_capline_keyid($capline)>.

=cut

sub get_capline_keyid($) {
	my ($capline) = @_;
	
	unless (defined $capline) {
		carp ('capline not defined');
		return undef;
	};
	
	my ($identifier, $address, $keyid, $version, $capabilities) = split /\s+/, $capline;
	$keyid = lc $keyid;
	unless ($keyid =~ /^[0-9a-f]{32}$/) {
		carp ("not a valid keyid '$keyid'");
		return undef;
	};
	return $keyid;
};


=item B<get_keyblock_keyid> (I<keyblock>)

Returns the keyid (in hex) of a mix2 public keyblock like
 -----Begin Mix Key-----
 21f7d8f2f12c29bf5d97a6d3435241b9
 258
 AATWvrxitS5QxQteau0tTtIlsLzG+InlABioGlpq
 4gW5btaabxJ6eu0I+FtSHSNd93NjNuCYIRxRrvsp
 //w846NMg0q9WzCeF1adsm6lKxxucIT/CE5/CMBH
 9rqravmnZg89ooefLSRn6IXEiEQudhz5B6hcqTn/
 BRyxEuYPkl8oxQAAAAAAAAAAAAAAAAAAAAAAAAAA
 AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
 AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
 AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
 AAAAAAAAAAAAAAAAAAAAAQAB
 -----End Mix Key-----

This is a "static" method, i.e. use it as
C<Mix::Remailer::get_keyblock_keyid($keyblock)> and not as
C<$remailer->get_keyblock_keyid($keyblock)>.

I<keyblock> should be a reference to an array;
=cut

sub get_keyblock_keyid($) {
	my ($keyblock) = @_;

	croak ('Keyblock not defined or not a reference to an array')
		if( ! defined($keyblock) || (ref($keyblock) ne 'ARRAY') );

	my @key = @{$keyblock};
	my $keyid = shift @key;

	unless ($keyid =~ /^[0-9a-f]{32}$/) {
		carp ('not a valid keyid');
		return undef;
	};
	return $keyid;
};

=pod

=back

=head1 AUTHOR

Peter Palfrader E<lt>peter@palfrader.orgE<gt>

=head1 BUGS
 
Please report them to the author.

=head1 SEE ALSO

=over

=item Mixmaster Protocol, Version 2 <draft-moeller-v2-01.txt>

=item mix(1)

=item news:alt.privacy.anon-server

=item Mix

=back

=cut
1;
