package Mix::V2Key;
# (c) 2002 Peter Palfrader <peter@palfrader.org>
# $Id: V2Key.pm,v 1.1 2002/05/21 02:22:05 weasel Exp $

# TODO cash keyid



=pod
=head1 NAME

Mix::V2Key - Mixmaster public or public and private key.

=head1 SYNOPSIS

FIXME docs

=head1 DESCRIPTION

This package provides basic key handling functionality for
Mix::Remailer.

=cut

use strict;
use lib '../../../../lib', 'lib', '../lib';
use Digest::MD5 qw{};
use Crypt::RSA;
use Carp qw{carp croak confess};
use Math::Pari qw(PARI);
use vars qw{$VERSION};

($VERSION) = '$Revision: 1.1 $' =~ /\s(\d+\.\d+)\s/;


=pod

=over

=item B<new> (%args)

The constructor.

=over

=item Keyblock (optional)

If %args has this key, set_public_form_keyblock is called with %args

=back

=cut

sub new {
	my ($class, %args) = @_;
	my $self = {};
	bless $self, $class;
	$self->set_public_from_keyblock( %args )
		if defined ($args{'Keyblock'});
	return $self;
};

# FIXME is there not better way?
# Stores a big unsigned integer in an array of octets
sub store_in_octets($$) {
	my ($len, $num) = @_;
	my @octets = map { 0 } (1 .. $len);
	my $index = $len - 1;
	while ($num > 0 && $index >= 0) {
		$octets[$index--] = $num % 256;
		$num = Math::Pari::floor($num / 256);
	}
	if ($num > 0) {
		croak('Number cannot be stored in that little octets.');
	};
	return @octets;
}

=item B<get_keyid> ()

Returns the key id of this mixmaster key in binary form.

=cut
sub get_keyid($) {
	my ($self) = @_;

	my $data = pack ("C128 C128",
		store_in_octets($self->{'Size'}, $self->{'Key'}->{'n'}),
		store_in_octets($self->{'Size'}, $self->{'Key'}->{'e'}) );
	return Digest::MD5::md5($data);
};


=item B<get_keyid_hex> ()

Returns the key id of this mixmaster key in hexadezimal form.

=cut
sub get_keyid_hex($) {
	my ($self) = @_;

	my $data = pack ("C128 C128",
		store_in_octets($self->{'Size'}, $self->{'Key'}->{'n'}),
		store_in_octets($self->{'Size'}, $self->{'Key'}->{'e'}) );
	return Digest::MD5::md5_hex($data);
};


=item B<set_public_from_keyblock> (%args)

<set_public_from_encoded> gets an encoded public mix key, parses
it and sets the object's public key attribute to it.

It returns the public keyid encdoed in hex.

One public mixkey looks as follows:

  [attributes list]
  
  -----Begin Mix Key-----
  [key ID]
  [length]
  [encoded key]
  -----End Mix Key-----

Example:
 marvin mix@marvin 21f7d8f2f12c29bf5d97a6d3435241b9 2.9b33 CNm

 -----Begin Mix Key-----
 21f7d8f2f12c29bf5d97a6d3435241b9
 258
 AATWvrxitS5QxQteau0tTtIlsLzG+InlABioGlpq
 4gW5btaabxJ6eu0I+FtSHSNd93NjNuCYIRxRrvsp
 //w846NMg0q9WzCeF1adsm6lKxxucIT/CE5/CMBH
 9rqravmnZg89ooefLSRn6IXEiEQudhz5B6hcqTn/
 BRyxEuYPkl8oxQAAAAAAAAAAAAAAAAAAAAAAAAAA
 AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
 AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
 AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
 AAAAAAAAAAAAAAAAAAAAAQAB
 -----End Mix Key-----

Keys of %args:

=over

=item B<Keyblock>

A referene to an array of lines of the public encoded key id.  Everything
between the C<-----Begin Mix Key-----> and C<-----End Mix Key-----> lines.

=back

=cut

sub set_public_from_keyblock($$) {
	my ($self, %args) = @_;

	if (! defined($args{'Keyblock'}) || ref($args{'Keyblock'}) ne 'ARRAY' ) {
		croak ('Keyblock not defined or not a reference to an array.');
		return undef;
	}
	
	my @key = @{$args{'Keyblock'}};
	my $keyid = shift @key;
	my $length = shift @key;
	my $base64_part = join '', @key;

	unless ($keyid =~ /^[0-9a-fA-F]{32}$/) {
		carp ("Invalid keyid in key '$keyid'\n");
		return undef;
	};
	unless ($length == 258) {
		# could be removed
		carp ("Unexpected length in key '$length' (!= 258)\n");
		return undef;
	};

	my $decoded = MIME::Base64::decode($base64_part);
	my $keylen;
	($keylen, $decoded) = unpack("v a*", $decoded);
	$length -= 2;
	if ($length % 2 != 0)  {
		carp ("Length must be even\n");
		return undef;
	};
	if ($keylen % 8 != 0 ) {
		carp ("keylen should be a multiple of 8\n");
		return undef;
	};
	if ($keylen != 8 * $length/2 ) {
		carp ("Given keysizes do not match ($keylen != 8 * $length/2 )\n");
		return undef;
	};
	if ($keylen != 1024)  {
		# could be removed
		carp ("Unexpected length in key '$keylen' (!= 1024)\n");
		return undef;
	};
	
	my $md5check = Digest::MD5::md5($decoded);
	unless ($keyid eq (unpack("H*", $md5check))) {
		carp ("Key does not match its keyid (checksum)\n");
		return undef;
	}
	my $exp = PARI(0);
	my $mod = PARI(0);
	my @octets = unpack ("C*", $decoded);
	if (scalar @octets != $length) {
		carp ("Given length does not match actual data length\n");
		return undef;
	};
	# FIXME quite inefficient
	for (my $i=0; $i<$length/2; $i++) {
		$mod *= 256;
		$mod += shift @octets;
	};
	for (my $i=0; $i<$length/2; $i++) {
		$exp *= 256;
		$exp += shift @octets;
	};
	if (scalar @octets) {
		carp ("Still data in \@octets (".scalar @octets." entries). Confused\n");
		return undef;
	};

	$self->{'Key'} = new Crypt::RSA::Key::Public ();
	$self->{'Key'}->{'n'} = $mod;
	$self->{'Key'}->{'e'} = $exp;
	$self->{'Size'} = $keylen / 8;

	return $keyid;
};

=pod

=back

=head1 AUTHOR

Peter Palfrader E<lt>peter@palfrader.orgE<gt>

=head1 BUGS
 
Please report them to the author.

=head1 SEE ALSO

Mixmaster Protocol, Version 2 <draft-moeller-v2-01.txt>,
mix(1),
news:alt.privacy.anon-server,
Mix

=cut
